%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Function name: find_plateau
%
% Scope:    define selected intervals for data analysis ("plateau")
%
% Input:    data (structure),
%           cutoff parameters (double),
%           list of gas type "flags" (i.e., MPV positions) (double),
%           list of gas names (cell),
%           plateau type index (double)
%
% Output:   updated data struture with defined plateau intervals for all gases, 
%           overview figures
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function data = find_plateau(data,cutoff,gastypeflag,gasnames,plateau_type_idx)

tic;fprintf('Running function find_plateau...\n');
  
cutoff=transpose(cutoff);
data.plateau=data.all;  

%% discard peaks in N2O standard deviation filter

std_perc_threshold=90;      %set percentile threshold for removing peaks in standard deviation 
std_avg_interval=31;        %set standard deviation moving average intervals (data points)

std_n2o = stdfilt(data.all(:,3),true(std_avg_interval));

for i=1:height(data.all)
    if std_n2o(i,1)>prctile(std_n2o,std_perc_threshold)
          data.plateau(i,:)=NaN;
    end
end
  
%% discard first 2 min after each multivalve position change

discard_idx=round(120/data.time_res);

for k=1:height(data.boundaries_idx)-1
      if data.boundaries_idx(k,1)+discard_idx>data.boundaries_idx(end,1)
          discard_idx=data.boundaries_idx(end,1)-data.boundaries_idx(k,1);
      end
      for i=data.boundaries_idx(k,1):data.boundaries_idx(k,1)+discard_idx
          data.plateau(i,:)=NaN;
      end 
end

%% find fixed duration plateaux (plateau_type_idx = 0)

if plateau_type_idx == 0   
    
    for k=1:height(data.boundaries_idx)-1
        for i=data.boundaries_idx(k,1):data.boundaries_idx(k+1,1)
            check(i-data.boundaries_idx(k,1)+1)=60*(transpose(max(data.plateau(i:data.boundaries_idx(k+1,1),1))-min(data.plateau(i:data.boundaries_idx(k+1,1),1))));
        end
        check_idx=find(check<cutoff(6,1));
        if ~isempty(check_idx)
            for i=1:height(data.all)
                if i<check_idx(1,1)+data.boundaries_idx(k,1) && i>data.boundaries_idx(k,1)
                    data.plateau(i,:)=NaN;
                end 
            end
        end
        if isempty(check_idx)
            for i=1:height(data.all)
                if i<data.boundaries_idx(k+1,1) && i>data.boundaries_idx(k,1)
                    data.plateau(i,:)=NaN;
                end 
            end
        end
    end
end

%% apply cut-off parameters (plateau_type_idx = 1)

if plateau_type_idx == 1 
    
    %apply cutoff n2o_std
    for k=1:height(data.boundaries_idx)-1
      for i=data.boundaries_idx(k,1):data.boundaries_idx(k+1,1)
            check(i-data.boundaries_idx(k,1)+1)=transpose(std(data.plateau(i:data.boundaries_idx(k+1,1),3),'omitnan'));
      end
      check_idx=find(check<cutoff(2,1));
      if ~isempty(check_idx)
          for i=1:height(data.all)
             if i<check_idx(1,1)+data.boundaries_idx(k,1) && i>data.boundaries_idx(k,1)
                data.plateau(i,:)=NaN;
             end 
          end
      end
      if isempty(check_idx)
          for i=1:height(data.all)
                if i<data.boundaries_idx(k+1,1) && i>data.boundaries_idx(k,1)
                    data.plateau(i,:)=NaN;
                end 
          end
      end
      clear check check_idx
    end

    %apply cutoff n2o_minmax
    for k=1:height(data.boundaries_idx)-1
      for i=data.boundaries_idx(k,1):data.boundaries_idx(k+1,1)
            check(i-data.boundaries_idx(k,1)+1)=transpose(max(data.plateau(i:data.boundaries_idx(k+1,1),3))-min(data.plateau(i:data.boundaries_idx(k+1,1),3)));
      end
      check_idx=find(check<cutoff(3,1));
      if ~isempty(check_idx)
        for i=1:height(data.all)
            if i<check_idx(1,1)+data.boundaries_idx(k,1) && i>data.boundaries_idx(k,1)
                data.plateau(i,:)=NaN;
            end 
        end
      end
      if isempty(check_idx)
          for i=1:height(data.all)
                if i<data.boundaries_idx(k+1,1) && i>data.boundaries_idx(k,1)
                    data.plateau(i,:)=NaN;
                end 
          end
      end
      clear check check_idx
    end

    %apply cutoff press_std
    for k=1:height(data.boundaries_idx)-1
      for i=data.boundaries_idx(k,1):data.boundaries_idx(k+1,1)
            check(i-data.boundaries_idx(k,1)+1)=transpose(std(data.plateau(i:data.boundaries_idx(k+1,1),8),'omitnan'));
      end
      check_idx=find(check<cutoff(4,1));
      if ~isempty(check_idx)
        for i=1:height(data.all)
            if i<check_idx(1,1)+data.boundaries_idx(k,1) && i>data.boundaries_idx(k,1)
                data.plateau(i,:)=NaN;
            end 
        end
      end
      if isempty(check_idx)
          for i=1:height(data.all)
                if i<data.boundaries_idx(k+1,1) && i>data.boundaries_idx(k,1)
                    data.plateau(i,:)=NaN;
                end 
          end
      end
      clear check check_idx
    end

    %apply cutoff press_minmax
    for k=1:height(data.boundaries_idx)-1
      for i=data.boundaries_idx(k,1):data.boundaries_idx(k+1,1)
            check(i-data.boundaries_idx(k,1)+1)=transpose(max(data.plateau(i:data.boundaries_idx(k+1,1),8))-min(data.plateau(i:data.boundaries_idx(k+1,1),8)));
      end
      check_idx=find(check<cutoff(5,1));
      if ~isempty(check_idx)
        for i=1:height(data.all)
            if i<check_idx(1,1)+data.boundaries_idx(k,1) && i>data.boundaries_idx(k,1)
                data.plateau(i,:)=NaN;
            end 
        end
      end
      if isempty(check_idx)
          for i=1:height(data.all)
                if i<data.boundaries_idx(k+1,1) && i>data.boundaries_idx(k,1)
                    data.plateau(i,:)=NaN;
                end 
          end
      end
      clear check check_idx
    end

    %apply cutoff min_duration
    for k=1:height(data.boundaries_idx)-1
          for i=data.boundaries_idx(k,1):data.boundaries_idx(k+1,1)
                check(i-data.boundaries_idx(k,1)+1)=transpose(max(data.plateau(i:data.boundaries_idx(k+1,1),2))-min(data.plateau(i:data.boundaries_idx(k+1,1),2)));
          end
          if check(1,1)<cutoff(1,1)/60
                for i=1:height(data.all)
                    if i<data.boundaries_idx(k+1,1) && i>data.boundaries_idx(k,1)
                        data.plateau(i,:)=NaN;
                    end 
                end
          end
          clear check check_idx
    end
end

%%  calculate duration of each interval

for k=1:height(data.boundaries_idx)-1
      for i=data.boundaries_idx(k,1):data.boundaries_idx(k+1,1)
            check(i-data.boundaries_idx(k,1)+1)=60*(transpose(max(data.plateau(i:data.boundaries_idx(k+1,1),2))-min(data.plateau(i:data.boundaries_idx(k+1,1),2))));
      end
      data.interval_duration(k,1)=check(1,1);
      clear check
end

  %% define gas types
  
  for k=1:width(gasnames)
        data.(gasnames{1,k})=NaN(height(data.plateau),width(data.plateau));
        for i=1:height(data.plateau)
             if data.all(i,12)==gastypeflag(1,k)
                  data.(gasnames{1,k})(i,:)=data.plateau(i,:); 
             end
        end
  end

%% plot plateau test figure 2  
    
figure;
subplot(2,1,1);
plot(data.all(:,2),data.all(:,3),'LineWidth',1.5,'Color',[0.7 0.7 0.7]); %,'handlevisibility','off');
hold on;
for i=1:width(gasnames)
plot(data.(gasnames{1,i})(:,2),data.(gasnames{1,i})(:,3),'LineWidth',1.5);
end
    for j=2:height(data.boundaries_idx)-1
        xline(data.all(data.boundaries_idx(j,1),2),'--k');
    end
grid on;
box on;
xlim([0 data.all(end,2)]);
xlabel('Time [h]','fontsize',12);
ylabel({'N_2O [ppb]'},'fontsize',12);
title('Plateau intervals definition','fontsize',12);
for i=1:width(gasnames)
lll{i}=string(gasnames{1,i});
end
leg=legend(['raw data',lll]);
leg.ItemTokenSize = [20,10];
leg.FontSize = 10;

subplot(2,1,2);
plot(data.all(:,2),data.all(:,12),'LineWidth',1.5,'Color','k','handlevisibility','off');
hold on;
    for j=2:height(data.boundaries_idx)-1
        xline(data.all(data.boundaries_idx(j,1),2),'--k');
    end
grid on;
box on;
xlim([0 data.all(end,2)]);
xlabel('Time [h]','fontsize',12);
ylabel({'MPV Position'},'fontsize',12);
title('MPV valve position','fontsize',12);

x0 = 10;
y0 = 50;
ww = 900;
hh = 700;
set(gcf,'units','points','position',[x0,y0,ww,hh]);
set(gcf,'Units','Inches');
pos = get(gcf,'Position');
set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3),pos(4)]);

  %% remove empty rows
  
  for k=1:width(gasnames)
      data.(gasnames{1,k})=rmmissing(data.(gasnames{1,k}),1,'MinNumMissing',width(data.all));
  end 

%%
time_elapsed=toc; fprintf('find_plateau completed (execution time: %1.2f s)\n',time_elapsed); 

end
